/**
* \file: challenge.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <limits.h>

#include "model/challenge.h"
#include "model/level_configuration.h"
#include "model/signature_db.h"
#include "control/configuration.h"
#include "encryption/signature.h"
#include "control/ald.h"

#include "util/logger.h"
#include "util/helper.h"

//---------------------------------------- local attributes ---------------------------------------------------------
static challenge_t current_challenge;
static bool challenge_valid=false;
static char chip_id_cache[CHALLENGE_CHIP_ID_SIZE];
static bool chip_id_available=false;
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- private function declaration ---------------------------------------------
static error_code_t challenge_get_chip_id(char *id_ptr);
static error_code_t challenge_read_out_chip_id(void);
static error_code_t challenge_extract_chip_id_from_line(char *line, bool *skip_line);

static void challenge_create_random_data(char *data, size_t data_size);

static error_code_t challenge_check_response_against_signature(const challenge_response_t *response);
static error_code_t challenge_check_response_against_signature_using_level_pubkey(const challenge_response_t *response,
		security_level_t level);
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- API members --------------------------------------------------------------
void challenge_create_new(const char *ecu_id)
{
	struct timeval time_stamp;
	//invalidate the current challenge
	challenge_invalidate();

	//create random data
	challenge_create_random_data(current_challenge.random_data, CHALLENGE_RANDOM_DATA_SIZE);

	//set the timestamp
	(void)gettimeofday(&time_stamp,NULL);
	current_challenge.tv_sec=(uint32_t)time_stamp.tv_sec;
	current_challenge.tv_usec=(uint32_t)time_stamp.tv_usec;

	//set the chip id
	if (challenge_get_chip_id(current_challenge.chip_id)!=RESULT_OK)
	{
		logger_log_error("CHALLENGE - Unable to determine a chip id. Using 0x0.");
		memset(current_challenge.chip_id,0, CHALLENGE_CHIP_ID_SIZE);
	}

	//set the ALD protocol major & minor
	current_challenge.ALD_protocol_major=ALD_PROTOCOL_MAJOR_VERSION;
	current_challenge.ALD_protocol_minor=ALD_PROTOCOL_MINOR_VERSION;

	//integrate the ecu_id
	memcpy(current_challenge.ecu_id, ecu_id, CHALLENGE_ECU_ID_SIZE);

	//key_set_id and reserved data are not used by now. Keep them as they are

	//validate the challenge
	challenge_valid=true;
}

void challenge_invalidate(void)
{
	if (challenge_valid)
		logger_log_debug("CHALLENGE - challenge now invalid.");
	challenge_valid=false;
	ald_deactivate_challenge_timeout();
}

const challenge_t *challenge_get_current(void)
{
	return &current_challenge;
}

error_code_t challenge_validate_response(const challenge_response_t *response)
{
	error_code_t result=RESULT_OK;

	if (!challenge_valid)
		result=RESULT_CHALLENGE_EXPIRED;

	if (result==RESULT_OK)
		result=challenge_check_response_against_signature(response);

	if (result==RESULT_OK)
	{
		//is the challenge the same we generated?
		if (memcmp(&response->challenge,&current_challenge,sizeof(challenge_t))!=0)
			result=RESULT_CHALLENGE_INVALID_RESPONSE;
	}

	return result;
}

void challenge_seed_random_number_generator(void)
{
	struct timeval curtime;
	gettimeofday(&curtime, NULL);
	srand((unsigned int)curtime.tv_usec);
}

bool challenge_is_valid(void)
{
	return challenge_valid;
}
//-------------------------------------------------------------------------------------------------------------------

//------------------------------------- private members -------------------------------------------------------------
static error_code_t challenge_get_chip_id(char *id_ptr)
{
	error_code_t result=RESULT_OK;

	if (!chip_id_available)
		result=challenge_read_out_chip_id();

	if (result==RESULT_OK)
		memcpy(id_ptr, chip_id_cache, CHALLENGE_CHIP_ID_SIZE);

	return result;
}

static error_code_t challenge_read_out_chip_id(void)
{
	FILE *file;
	error_code_t result=RESULT_OK;
	char *line=NULL;
	size_t line_buf_size=0;
	bool searching=true;

	file=fopen("/proc/cpuinfo","r");
	if (file==NULL)
	{
		logger_log_error("Unable to read out the chip id. The file \"/proc/cpuinfo\" not available.");
		result=RESULT_NORESOURCE;
	}

	while(searching && result==RESULT_OK)
	{
		if (getline(&line, &line_buf_size,file)==-1)
		{
			searching=false;
			continue;
		}

		result=challenge_extract_chip_id_from_line(line, &searching);
	}

	if (line!=NULL)
		free(line);

	if (file !=NULL)
		fclose(file);

	if (result==RESULT_OK && !chip_id_available)
	{
		logger_log_error("Unable to read out the chip id. The tag \"%s\" could not be "
				"found in file \"/proc/cpuinfo\".", PROC_CHIP_ID_KEY);
		result=RESULT_NORESOURCE;
	}

	return result;
}

static error_code_t challenge_extract_chip_id_from_line(char *line, bool *skip_line)
{
	bool is_empty=false;
	char *clean_line;
	char *value;
	char *parse_result_ptr;
	uint64_t chip_id;

	clean_line=helper_trim(line, &is_empty);

	//emtpy? -> skip the line
	if (is_empty)
	{
		*skip_line=true;
		return RESULT_OK;
	}

	//Do we start with the wrong tag in the line? -> skip the line
	if (strstr(clean_line, PROC_CHIP_ID_KEY)!=clean_line)
	{
		*skip_line=true;
		return RESULT_OK;
	}

	value=strchr(clean_line, ':');
	if (value==NULL)
	{
		logger_log_error("Unable to read out the chip id. Expecting line \"%s : <id>\" in "
				"file \"/proc/cpuinfo\".", PROC_CHIP_ID_KEY);
		return RESULT_NORESOURCE;
	}

	value++;

#if (CHALLENGE_CHIP_ID_SIZE != 8)
#error This implementation works only for a chip id size of 8. For other sizes you need to adapt
#endif

	chip_id=strtol(value, &parse_result_ptr, 16);
	if (value==parse_result_ptr)
	{
		logger_log_error("Unable to read out the chip id. Expecting an 8 bytes hex coded number after \"%s\" in "
				"file \"/proc/cpuinfo\".", PROC_CHIP_ID_KEY);
		return RESULT_NORESOURCE;
	}

	memcpy(chip_id_cache, &chip_id,8);
	chip_id_available=true;
	*skip_line=false;

	return RESULT_OK;
}

static void challenge_create_random_data(char *data, size_t data_size)
{
	uint32_t i;
	uint32_t bytes_of_number_remaining=0;
	unsigned int random_number=0;

	for (i=0;i<data_size;i++)
	{
		//get a new random number if we completely used the old one
		if (bytes_of_number_remaining==0)
		{
			random_number=(unsigned int)rand();
			bytes_of_number_remaining=sizeof(random_number);
		}
		//copy the first 8 bits of the random number to the current data byte
		data[i]=(char)(unsigned char)(0xFF & random_number);
		//shift out the used part of the random number
		random_number >>=8;
		//increment the number of unused bits of the random number
		bytes_of_number_remaining--;
	}
}

static error_code_t challenge_check_response_against_signature(const challenge_response_t *response)
{
	level_list_item_t *level_list_item;

	level_list_item=level_configuration_get_list_item(response->targeted_level);
	if (level_list_item==NULL)
		return RESULT_INVALID_LEVEL;

	while (level_list_item!=NULL)
	{
		security_level_t level=level_configuration_list_item_get_level(level_list_item);
		if (level_configuration_list_item_has_level_pubkey(level_list_item))
		{
			error_code_t result;
			logger_log_debug("CHALLENGE - Decrypting the signature with the public key found in level %d.",level);
			result=challenge_check_response_against_signature_using_level_pubkey(response,level);
			if (result==RESULT_OK)
			{
				logger_log_info("The private key of level %d was used to initiate the level change.",level);
				logger_log_errmem("ALD - The private key of level %d was used to initiate the level change.",level);
				logger_log_debug("CHALLENGE - Challenge solved correctly.");
				return RESULT_OK;
			}
			else
			{
				if (configuration_get_challenge_verify_key_behaviour() == UNLOCK_ONLY_THAT_LEVEL)
				{
					logger_log_info("CHALLENGE - The public key does not solve the challenge correctly."
									" Incorrect private key is used to initiate level change.");
					return RESULT_CHALLENGE_INVALID_RESPONSE;
				}
				else
					logger_log_debug("CHALLENGE - The public key does not solve the challenge correctly."
							" Trying with the key of the next higher level.");
			}
		}
		else
		{
			if (configuration_get_challenge_verify_key_behaviour() == UNLOCK_ONLY_THAT_LEVEL)
			{
				logger_log_error("CHALLENGE - Level %d has no public key."
						"Also The keys are configured to be valid for only one level", level);
				logger_log_errmem("ALD - Configuration Error - Level %d not reachable!", level);
				return RESULT_INVALID_LEVEL_CONFIGURATION;
			}
			else
			{
				logger_log_error("CHALLENGE - Level %d has no public key."
						"Checking the next higher level for a key.",level);
				logger_log_debug("Level %d has no public key,Checking the next higher level for a key",level);
			}
		}
		level_list_item=level_configuration_get_next_list_item(level_list_item);
	}

	return RESULT_CHALLENGE_INVALID_RESPONSE;
}

static error_code_t challenge_check_response_against_signature_using_level_pubkey(const challenge_response_t *response,
		security_level_t level)
{
	EVP_PKEY *pubkey;
	char key_path[PATH_MAX];
	const char *root_dir;
	error_code_t result;

	root_dir=configuration_get_script_root_dir();
	level_configuration_get_key_path_of_level(key_path,PATH_MAX,root_dir,level);

	logger_log_debug("CHALLENGE - Validating public key for level %d: %s",level,key_path);
	if (!signature_db_validate_script(key_path))
	{
		logger_log_error("Public key of level %d modified. It is not used to validate the response of the request.",level);
		return RESULT_VERIFICATION_KEY_MODIFIED;
	}

	logger_log_debug("CHALLENGE - Reading in public key for level %d: %s",level,key_path);
	if (signature_read_public_key(key_path,&pubkey, false)!=RESULT_OK)
	{
		logger_log_error("Unable to find the public key of level %d.",level);
		return RESULT_VERIFICATION_FAILED;
	}
	result=signature_check_data_block(response, sizeof(challenge_response_t)-RESPONSE_SIGNATURE_SIZE_MAX,response->signature,pubkey);

	signature_destroy_key(pubkey);
	return result;
}
//-------------------------------------------------------------------------------------------------------------------
